var RichPlugins = RichPlugins || {

    Instances : {
        Lists   : {},
        Tags    : {},
        Grids   : {},
        Sliders : {},
        Common  : {}
    },

    Utils: {

        __: function(text, trans) {
            return (trans && trans[text]) || text;
        },

        init: function(el, opts) {
            let timeEls  = el.getElementsByClassName('rpi-time'),
                textEls  = el.getElementsByClassName('rpi-text');

            for (let i = 0; i < timeEls.length; i++) {
                timeEls[i].innerHTML = this.time(timeEls[i].getAttribute('data-time'), opts.time_format);
            }
            for (let i = 0; i < textEls.length; i++) {
                let textEl = textEls[i];
                if (textEl && textEl.innerHTML) {
                    textEl.innerHTML = this.trimtext(textEl.innerHTML, opts.text_size, opts.trans);
                    var readmoreEl = textEl.getElementsByClassName('rpi-readmore')[0];
                    if (readmoreEl) {
                        readmoreEl.onclick = RichPlugins.Utils.opentext;
                    }
                }
            }
        },

        ajax: function(url, cb) {
            const xhr = new XMLHttpRequest();
            xhr.open('POST', url, true);
            xhr.setRequestHeader('Content-Type', 'application/json');
            xhr.onreadystatechange = function() {
              if ((xhr.readyState === XMLHttpRequest.DONE || xhr.readyState === xhr.DONE) && xhr.status === 200) {
                cb(JSON.parse(xhr.responseText));
              }
            };
            xhr.send();
        },

        stars: function(rating) {
            var str = '';
            for (var i = 1; i < 6; i++) {
                var score = rating - i;
                if (score >= 0) {
                    str += '<span class="rpi-star"></span>';
                } else if (score > -1 && score < 0) {
                    if (score < -0.75) {
                        str += '<span class="rpi-star rpi-star-o"></span>';
                    } else if (score > -0.25) {
                        str += '<span class="rpi-star"></span>';
                    } else {
                        str += '<span class="rpi-star rpi-star-h"></span>';
                    }
                } else {
                    str += '<span class="rpi-star rpi-star-o"></span>';
                }
            }
            return str;
        },

        time: function(time, format) {
            return format ? time : WPacTime.getTimeAgo(parseInt(time) * 1000, _rplg_lang());
        },

        trimtext: function(text, size, trans) {
            if (text && size && text.length > size) {
                var subtext = text.substring(0, size),
                    idx = subtext.indexOf(' ') + 1;

                if (idx < 1 || size - idx > (size / 2)) {
                    idx = size;
                }

                var vtext = '', invtext = '';
                if (idx > 0) {
                    vtext = text.substring(0, idx - 1);
                    invtext = text.substring(idx - 1, text.length);
                }

                return vtext + (
                    invtext ?
                        '<span class="rpi-s">... </span>' +
                        '<input type="hidden" value="' + invtext + '"></input>' +
                        '<span class="rpi-readmore">' + this.__('read more', trans) + '</span>' : ''
                    );
            } else {
                return text;
            }
        },

        opentext: function() {
            let textEl = this.parentNode,
                hide = this.previousSibling,
                dots = this.previousSibling.previousSibling,
                txt = hide.value;
            RichPlugins.Utils.rm(hide);
            RichPlugins.Utils.rm(dots);
            RichPlugins.Utils.rm(this);
            textEl.innerHTML += txt;
        },

        anchor: function(url, text, opts) {
            let rel = [];
            if (opts.open_link) {
                rel.push('noopener');
            }
            if (opts.nofollow_link) {
                rel.push('nofollow');
            }
            rel = rel.length ? 'rel="' + rel.join(' ') + '"' : '';
            return '<a href="' + url + '" ' + (opts.open_link ? 'target="_blank"' : '') + ' ' + rel + '>' + text + '</a>';
        },

        rm: function(el) {
            el && el.parentNode && el.parentNode.removeChild(el);
        },

        brsCompare: function(a, b) {
            return parseInt(a.split(':')[0]) > parseInt(b.split(':')[0]) ? 1 : -1;
        },

        reviewInit: function(reviewEl, options) {
            let timeEl  = reviewEl.getElementsByClassName('rpi-time')[0],
                textEl  = reviewEl.getElementsByClassName('rpi-text')[0];

            if (timeEl) {
                timeEl.innerHTML = RichPlugins.Utils.time(timeEl.getAttribute('data-time'), options.time_format);
            }

            if (textEl && textEl.innerHTML) {
                textEl.innerHTML = RichPlugins.Utils.trimtext(textEl.innerHTML, options.text_size, options.trans);
                var readmoreEl = textEl.getElementsByClassName('rpi-readmore')[0];
                if (readmoreEl) {
                    readmoreEl.onclick = RichPlugins.Utils.opentext;
                }
            }
        },

        starsInit: function(starsEl) {
            let starsInfo = starsEl.getAttribute('data-info').split(',');
            starsEl.innerHTML = render_stars(starsInfo[0], starsInfo[1], starsInfo[2]);
        },

        setCol: function(col) {
            let el = this.getStyle();
            el.innerHTML = el.innerHTML.replace(
                /$|(\.rpi-card\{--col:\d+!important\})/,
                '.rpi-card{--col:' + col + '!important;--gap:' + (col-1) + '!important}'
            );
        },

        getCol: function() {
            let el = this.getStyle(),
                match = el.innerHTML.match(/--col:(\d+)/);
            return match && match.length > 1 ? match[1] : !1;
        },

        getStyle: function() {
            let el = document.getElementById('rpi-style');
            if (!el) {
                el = document.createElement('style');
                el.id = 'rpi-style';
                document.head.appendChild(el);
            }
            return el;
        },

        capit: function(s) {
            return s.charAt(0).toUpperCase() + s.slice(1);
        }
    },

    Common: function(rootEl, options) {

        const TIMEOUT_INIT = 100,

            collId  = rootEl.getAttribute('data-id'),
            cntEl   = rootEl.getElementsByClassName('rpi-cnt')[0],
            headerEl = rootEl.getElementsByClassName('rpi-header')[0];
            contentEl = rootEl.getElementsByClassName('rpi-content')[0];

        var This = null;

        return This = {

            isVisible: function(el) {
                return !!(el.offsetWidth || el.offsetHeight || el.getClientRects().length) && window.getComputedStyle(el).visibility !== 'hidden';
            },

            resize: function() {
                let size = This.getSize(),
                    offsetWidth = cntEl.offsetWidth;

                cntEl.className = cntEl.className.replace(/rpi-col-[xsml]+/, 'rpi-col-' + size);

                if (options.breakpoints) {
                    let brs = options.breakpoints.split(',');
                    brs.sort(RichPlugins.Utils.brsCompare);

                    for (let i = 0; i < brs.length; i++) {
                        let vals = brs[i].split(':');
                        if (offsetWidth < parseInt(vals[0])) {
                            RichPlugins.Utils.setCol(vals[1]);
                            break;
                        }
                    }
                }
                return size;
            },

            getSize: function() {
                let size, offsetWidth = cntEl.offsetWidth;
                if (offsetWidth < 510) {
                    size = 'xs';
                } else if (offsetWidth < 750) {
                    size = 'x';
                } else if (offsetWidth < 1100) {
                    size = 's';
                } else if (offsetWidth < 1450) {
                    size = 'm';
                } else if (offsetWidth < 1800) {
                    size = 'l';
                } else {
                    size = 'xl';
                }
                return size;
            },

            loadNextReviews: function(size, cb) {
                let count = parseInt(cntEl.getAttribute('data-count')),
                    offset = parseInt(cntEl.getAttribute('data-offset'));
                if (size > 0) {
                    let list = [];
                    This.preloadReviews(list, count, offset, size);
                    This.loadAjaxReviews(list, offset, size, cb);
                }
            },

            preloadReviews: function(list, count, offset, size) {
                let preloadSize = count - offset > size ? size : count - offset;
                    reviewsEl = contentEl.getElementsByClassName('rpi-cards')[0],
                    reviewsList = reviewsEl.getElementsByClassName('rpi-card'),
                    len = reviewsList.length - 1;

                cntEl.setAttribute('data-offset', offset + preloadSize);
                for (let i = 0; i < preloadSize; i++) {
                    let randReviewEl = reviewsList[Math.round(Math.random() * len)],
                        cloneEl = randReviewEl.cloneNode(true);
                    cloneEl.style = 'filter: blur(4px);';
                    reviewsEl.appendChild(cloneEl);
                    list.push(cloneEl);
                }
            },

            loadAjaxReviews: function(list, offset, size, cb) {
                let url  = brb_vars.ajaxurl + '?action=brb_get_reviews&id=' + collId + '&offset=' + offset + '&size=' + size;
                RichPlugins.Utils.ajax(url, function(json) {
                    let len = json.reviews.length,
                        options = JSON.parse(rootEl.getAttribute('data-opts'));
                    for (let i = 0; i < len; i++) {
                        let converted = This.convertReviewEl(list.shift(), json.reviews[i]);
                        RichPlugins.Utils.reviewInit(converted, options);
                        This.initReviewStyle(converted, options);
                        This.initLogoStyle(converted, options.style_logo);
                    }
                    while(list.length) {
                        RichPlugins.Utils.rm(list.shift());
                    }

                    // In case if AJAX returns wrong number of reviews (very rare: server timeout or cache issues)
                    if (offset + size != offset + len) {
                        cntEl.setAttribute('data-offset', offset + len);
                    }

                    if (cb) cb();
                });
            },

            convertReviewEl: function(el, review) {
                el.innerHTML = This.reviewInner(review);
                el.style = '';
                el.setAttribute('data-provider', review.provider);
                return el;
            },

            reviewInner: function(review) {
                let img = review.author_avatar,
                    avasize = rootEl.style.getPropertyValue('--img-size'),
                    imgsize = avasize ? avasize : 56,
                    url = review.author_url ? review.author_url : '';
                return '' +
                    '<div class="rpi-card-inner">' +
                        '<i></i>' +
                        '<div class="rpi-flex">' +
                            (options.hide_avatar ? '' : This.reviewAvatar(review)) +
                            '<div class="rpi-info">' +
                                (options.hide_name ? '' : This.reviewName(review)) +
                                (options.disable_review_time ? '' : '<div class="rpi-time" data-time="' + review.time + '"></div>') +
                                '<div class="rpi-stars" data-info="' + [review.rating,review.provider,].join(',') + '">' +
                                    RichPlugins.Utils.stars(review.rating) +
                                '</div>' +
                            '</div>' +
                        '</div>' +
                        '<div class="rpi-body">' +
                            (review.text || review.media || review.reply ? '<div class="rpi-text rpi-scroll">' + review.text + '</div>' : '') +
                            (review.media ? '<div class="rpi-media">' + This.reviewMedia(review) + '</div>' : '') +
                            (review.reply ? '<div class="rpi-reply rpi-scroll">' +
                                                '<b>' + RichPlugins.Utils.__('Response from the owner', options.trans) + ':</b> ' + review.reply +
                                            '</div>' : '') +
                        '</div>' +
                    '</div>';
            },

            reviewReply: function(reviewEl, reply) {
                let replyEl = reviewEl.getElementsByClassName('rpi-reply')[0];
                if (!replyEl) {
                    replyEl = document.createElement('div');
                    replyEl.className = 'rpi-reply rpi-scroll';
                    let bodyEl = reviewEl.getElementsByClassName('rpi-body')[0];
                    bodyEl.appendChild(replyEl);
                }
                replyEl.innerHTML = '<b>' + RichPlugins.Utils.__('Response from the owner', options.trans) + ':</b> ' + reply;
            },

            reviewName: function(review) {
                let name = review.author_name ? review.author_name : RichPlugins.Utils.capit(review.provider) + ' user';
                return '' +
                    '<div class="rpi-name" title="' + name + '">' +
                        (review.author_url && !options.disable_user_link ? RichPlugins.Utils.anchor(review.author_url, name, options) : name) +
                    '</div>';
            },

            reviewAvatar: function(review) {
                let imgurl = review.author_avatar,
                    avasize = rootEl.style.getPropertyValue('--img-size'),
                    imgsize = avasize ? avasize.replace('px', '') : 56,
                    name = review.author_name ? review.author_name : RichPlugins.Utils.capit(review.provider) + ' user';
                switch (review.provider) {
                    case 'google':
                        let matches = imgurl.match(/googleusercontent\.com\/([^\/]+)\/([^\/]+)\/([^\/]+)\/([^\/]+)\/photo\.jpg/);
                        if (matches && matches.length > 4 && matches[3][0] == 'AAAAAAAAAAA') {
                            imgurl = imgurl.replace('/photo.jpg', '/s128-c0x00000000-cc-rp-mo/photo.jpg');
                        }
                        if (imgurl) {
                            if (imgurl.indexOf('s128') > -1) {
                                imgurl = imgurl.replace('s128', 's' + imgsize);
                            } else if (imgurl.indexOf('-mo') > -1) {
                                imgurl = imgurl.replace('-mo', '-mo-s' + imgsize);
                            } else {
                                imgurl = imgurl.replace('-rp', '-rp-s' + imgsize);
                            }
                        }
                        break;
                    case 'yelp':
                        if (imgurl) {
                            imgurl = imgurl.replace(/(ms)|(o)\.jpg/, (imgsize <= 128 ? 'ms' : 'o') + '.jpg');
                        }
                        break;
                }
                return '' +
                    '<div class="rpi-img">' +
                        '<img src="' + imgurl + '" class="rpi-img" ' + (options.lazy_load_img ? 'loading="lazy"' : '') +
                        ' alt="' + name + '" width="' + imgsize + '" height="' + imgsize + '" title="' + name + '"' +
                        ' onerror="if(this.src!=\'http://test.richplugins.com/wp-content/plugins/business-reviews-bundle/assets/img/google_avatar.png\')' +
                                 'this.src=\'http://test.richplugins.com/wp-content/plugins/business-reviews-bundle/assets/img/google_avatar.png\';">' +
                    '</div>';
            },

            reviewMedia: function(review) {
                let thumbs = '';
                for (let i = 0; i < review.media.length; i++) {
                    thumbs += '<div onclick="_rplg_popup(\'' + review.media[i].googleUrl + '\', 800, 600)" ' +
                             'style="background-image:url(' + review.media[i].thumbnailUrl.replace('=s300', '=s50') + ')" ' +
                             'class="rpi-thumb rpi-clickable"></div>'
                }
                return thumbs;
            },

            initStyle: function() {
                let options = JSON.parse(rootEl.getAttribute('data-opts'));

                if (headerEl) {
                    let cards = headerEl.getElementsByClassName('rpi-card');
                    for (let j = 0; j < cards.length; j++) {
                        this.initLogoStyle(cards[j], options.head_logo);
                        this.initYelpBasedLogo(cards[j], options.head_logo);
                    }
                }
                if (contentEl) {
                    let reviewsEl = contentEl.getElementsByClassName('rpi-cards')[0],
                        reviewCards = reviewsEl.getElementsByClassName('rpi-card');
                    for (let j = 0; j < reviewCards.length; j++) {
                        this.initReviewStyle(reviewCards[j], options);
                    }
                    this.initCtrlStyle(options);

                    let cards = contentEl.getElementsByClassName('rpi-card');
                    for (let j = 0; j < cards.length; j++) {
                        this.initLogoStyle(cards[j], options.style_logo);
                    }
                }
            },

            initYelpBasedLogo: function(card, style_logo) {
                if (style_logo == 5) {
                    let yelpLogo = card.getElementsByClassName('rpi-logo-yelp')[0];
                    if (yelpLogo) {
                        yelpLogo.classList.remove('rpi-logo');
                        yelpLogo.classList.add('rpi-logo_after');
                        yelpLogo.innerText = 'yelp';
                    }
                }
            },

            initReviewStyle: function(reviewEl, options) {
                let R   = reviewEl,
                    RI  = R.getElementsByClassName('rpi-card-inner')[0],
                    i   = R.getElementsByTagName('i')[0],
                    rf  = R.getElementsByClassName('rpi-flex')[0],
                    rim = R.getElementsByClassName('rpi-img')[0],
                    rii = R.getElementsByClassName('rpi-info')[0],
                    rt  = R.getElementsByClassName('rpi-time')[0],
                    rn  = R.getElementsByClassName('rpi-name')[0],
                    rs  = R.getElementsByClassName('rpi-stars')[0],
                    rb  = R.getElementsByClassName('rpi-body')[0],
                    rtt = R.getElementsByClassName('rpi-text')[0],
                    rr  = R.getElementsByClassName('rpi-reply')[0];

                if (rn) rii.appendChild(rn);
                if (rt) rii.appendChild(rt);
                if (rs) rii.appendChild(rs);

                R.className  = 'rpi-card';
                RI.className = 'rpi-card-inner';
                rf.className = 'rpi-flex';
                rb.className = 'rpi-body';
                i.className  = '';

                switch(options.style_style) {

                    case '1':
                    case '7':
                        RI.appendChild(rf);
                        if (rb) RI.appendChild(rb);
                        rb.className += ' rpi-normal-up__body';
                        break;

                    case '2':
                    case '8':
                        if (rb) RI.appendChild(rb);
                        RI.appendChild(rf);
                        rb.className += ' rpi-normal-down__body';
                        break;

                    case '3':
                        R.appendChild(rf);
                        R.appendChild(RI);
                        if (rb) RI.appendChild(rb);

                        i.className = 'rpi-triangle rpi-triangle_up';
                        rf.className += ' rpi-bubble__flex rpi-bubble__flex_up';

                        if (this.getCssVar('--card-br') != 'none') {
                            i.className += ' rpi-triangle__br_up';
                            RI.className += ' rpi-bubble__inner rpi-bubble__inner_up';
                        }
                        break;

                    case '4':
                        R.appendChild(RI);
                        R.appendChild(rf);
                        if (rb) RI.appendChild(rb);

                        i.className = 'rpi-triangle rpi-triangle_down';
                        rf.className += ' rpi-bubble__flex rpi-bubble__flex_down';

                        if (this.getCssVar('--card-br') != 'none') {
                            i.className += ' rpi-triangle__br_down';
                            RI.className += ' rpi-bubble__inner rpi-bubble__inner_down';
                        }
                        break;

                    case '5':
                        RI.appendChild(rf);
                        rf.appendChild(rii);
                        if (rb) rii.appendChild(rb);
                        rf.className += ' rpi-shift-up__flex';
                        break;

                    case '6':
                        RI.appendChild(rf);
                        rf.appendChild(rii);
                        if (rb) rii.prepend(rb);
                        rf.className += ' rpi-shift-down__flex';
                        break;
                }
                switch(options.style_style) {
                    case '7':
                        R.className += ' rpi-avaborder-up';
                        break;
                    case '8':
                        R.className += ' rpi-avaborder-down';
                        break;
                }

                if (rb && rr) rb.appendChild(rr);

                if (rs) {
                    switch (options.style_stars) {

                        case '1':
                            if (rt) {
                                rii.insertBefore(rs, rt);
                                rii.insertBefore(rt, rs);
                            } else if (rn) {
                                rii.insertBefore(rs, rn);
                                rii.insertBefore(rn, rs);
                            } else {
                                rii.appendChild(rs);
                            }
                            break;

                        case '2':
                            if (rt) rii.insertBefore(rs, rt);
                            else rii.appendChild(rs);
                            break;

                        case '3':
                            if (rn) {
                                rii.insertBefore(rs, rn);
                            } else if (rt) {
                                rii.insertBefore(rs, rt);
                            } else {
                                rii.prepend(rs);
                            }
                            break;

                        case '4':
                            if (rb) rb.prepend(rs);
                            else RI.appendChild(rs);
                            break;

                        case '5':
                            if (rtt) rtt.prepend(rs);
                            else rb.appendChild(rs);
                            break;

                        case '6':
                            if (rb) {
                                rb.appendChild(rs);
                                if (rr) rb.insertBefore(rs, rr);
                            } else RI.appendChild(rs);
                            break;
                    }
                }

                if (rim) {
                    switch (options.style_ava) {
                        case '1':
                            rf.appendChild(rim);
                            rf.appendChild(rii);
                            break;
                        case '2':
                            rf.appendChild(rii);
                            rf.appendChild(rim);
                            break;
                    }
                }
            },

            initCtrlStyle: function(options) {
                let ctrl = contentEl.getElementsByClassName('rpi-controls')[0];
                if (ctrl) {
                    ctrl.className = 'rpi-controls';
                    switch(options.style_style) {
                        case '3':
                            ctrl.className += ' rpi-bubble-up__ctrl';
                            break;
                        case '4':
                            ctrl.className += ' rpi-bubble-down__ctrl';
                            break;
                    }
                }
            },

            initLogoStyle: function(cardEl, style_logo) {
                let cardInEl = cardEl.getElementsByClassName('rpi-card-inner')[0],
                    provider = cardEl.getAttribute('data-provider'),
                    platforms = provider.split(',');

                if (platforms.length > 1) {
                    this.initLogoMultiple(cardEl, cardInEl, platforms, style_logo);
                } else {
                    this.initLogoSingle(cardEl, cardInEl, provider, style_logo);
                }
            },

            initLogoSingle: function(cardEl, cardInEl, provider, style_logo) {
                let logoEl = cardEl.getElementsByClassName('rpi-logo')[0],
                    cls = ['rpi-logo', 'rpi-logo-' + provider];

                if (logoEl) {
                    logoEl.classList.remove(...cls);
                }

                switch (style_logo) {
                    case '1':
                    case '6':
                        let body = cardEl.getElementsByClassName('rpi-body')[0];
                        if (style_logo == '6') {
                            cls.push('rpi-logo-left');
                        }
                        if (cardInEl.classList.contains('rpi-bubble__inner') && body) {
                            body.classList.add(...cls);
                        } else {
                            cardInEl.classList.add(...cls);
                        }
                        break;
                    case '2':
                        let nameEl = cardEl.getElementsByClassName('rpi-name')[0];
                        if (nameEl) {
                            nameEl.classList.add(...cls);
                        }
                        break;
                    case '3':
                        let imgEl = cardEl.getElementsByClassName('rpi-img')[0];
                        imgEl.classList.add(...cls);
                        break;
                    case '4':
                        let starsEl = cardEl.getElementsByClassName('rpi-stars')[0];
                        starsEl.classList.add(...cls);
                        break;
                    case '5':
                        let basedEl = cardEl.getElementsByClassName('rpi-based')[0];
                        if (basedEl) {
                            let count = basedEl.innerText.match(/(\d+)/)[1];
                            basedEl.innerHTML = basedEl.innerText.replace(count, count + ' <span class="' + cls.join(' ') + '"></span>');
                        }
                        break;
                }
            },

            initLogoMultiple: function(cardEl, cardInEl, platforms, style_logo) {
                let logosEl = cardEl.getElementsByClassName('rpi-logos')[0];

                if (!logosEl) {
                    logosEl = this.createLogos(platforms);
                }

                switch (style_logo) {
                    case '0':
                        RichPlugins.Utils.rm(logosEl);
                        break;
                    case '1':
                        cardInEl.prepend(logosEl);
                        break;
                    case '2':
                        let nameEl = cardEl.getElementsByClassName('rpi-name')[0];
                        if (nameEl) {
                            nameEl.prepend(logosEl);
                        }
                        break;
                    case '4':
                        let scoreEl = cardEl.getElementsByClassName('rpi-score')[0],
                            starsEl = cardEl.getElementsByClassName('rpi-stars')[0];
                        scoreEl.appendChild(logosEl);
                        scoreEl.insertBefore(logosEl, starsEl);
                        break;
                    case '5':
                        let basedEl = cardEl.getElementsByClassName('rpi-based')[0];
                        if (basedEl) {
                            let count = basedEl.innerText.match(/(\d+)/)[1];
                            basedEl.innerHTML = basedEl.innerText.replace(count, count + ' ' + logosEl.outerHTML);
                            RichPlugins.Utils.rm(logosEl);
                        }
                        break;
                }
            },

            createLogos: function(platforms) {
                let logosEl = document.createElement('span');
                logosEl.className = 'rpi-logos';
                for (let i = 0; i < platforms.length; i++) {
                    logosEl.innerHTML += '<span class="rpi-logo rpi-logo-' + platforms[i] + '">';
                }
                return logosEl;
            },

            getCssVar: function(name) {
                return rootEl.style.getPropertyValue(name);
            }
        }
    },

    /**
     * Tag layout
     */
    Tag: function(rootEl, options) {

        const collId = rootEl.getAttribute('data-id'),
            Common   = RichPlugins.Common(rootEl, options)
            cntEl    = rootEl.getElementsByClassName('rpi-tag-cnt')[0],
            tagOpts  = JSON.parse(cntEl.getAttribute('data-opts'));

        var This = RichPlugins.Instances.Tags[collId];

        return This = {

            init: function() {
                let starsReviewUsEl = rootEl.querySelector('.rpi-stars[data-reviewus]');
                if (starsReviewUsEl) {
                    RichPlugins.Utils.starsInit(starsReviewUsEl);
                    starsReviewUsEl.onclick = function(e) {
                        var svg = e.target.tagName == 'svg' ? e.target : e.target.parentNode,
                            idx = [...svg.parentNode.children].indexOf(svg);
                        _rplg_popup(idx > 2 ? this.getAttribute('data-reviewus') : 'https://app.richplugins.com/feedback?s=' + idx, 800, 600);
                    };
                }

                if (tagOpts.tag_popup > 0) {
                    setTimeout(function() {
                        rootEl.className += ' rpi-pop-up';
                    }, tagOpts.tag_popup * 1000);
                }

                if (tagOpts.tag_click == 'sidebar') {

                    let sbId = 'rpi-tag-sidebar-' + collId,
                        sbRootEl = document.getElementById(sbId);

                    if (!sbRootEl) {
                        sbRootEl = document.createElement('div');
                        sbRootEl.id = sbId;
                        sbRootEl.className = 'rpi';
                        sbRootEl.innerHTML = '' +
                            '<div class="rpi-sb" style="display:none">' +
                                '<div class="rpi-sbb"></div>' +
                                '<div class="rpi-sbc">' +
                                    '<div class="rpi-sbci"></div>' +
                                '</div>' +
                                '<div class="rpi-sbx">×</div>' +
                            '</div>';
                        document.body.appendChild(sbRootEl);
                    }

                    let sbEl = sbRootEl.getElementsByClassName('rpi-sb')[0],
                        sbxEl = sbEl.getElementsByClassName('rpi-sbx')[0];

                    sbxEl.onclick = function(e) {
                        sbEl.style.display = sbEl.style.display == 'none' ? 'block' : 'none';
                    };

                    rootEl.onclick = function(e) {
                        sbEl.style.display = sbEl.style.display == 'none' ? 'block' : 'none';

                        let sbciEl = sbEl.getElementsByClassName('rpi-sbci')[0];
                        if (sbciEl.innerHTML == '') {
                            let url  = brb_vars.ajaxurl + '?action=brb_embed&brb_collection_id=' + collId + '&brb_view_mode=' + tagOpts.tag_sidebar;
                            RichPlugins.Utils.ajax(url, function(json) {
                                sbciEl.innerHTML = json.data;
                                RichPlugins.init(sbciEl.querySelector('.rpi[data-exec=""]'));
                            });
                        }
                    };
                }

                RichPlugins.Instances.Tags[collId] = This;
                console.log('RichPlugins Tag initialized');
            }
        }
    },

    /**
     * List layout
     */
    List: function(rootEl, options) {

        const collId  = rootEl.getAttribute('data-id'),
            cntEl = rootEl.getElementsByClassName('rpi-cnt')[0];

        var This = RichPlugins.Instances.Lists[collId],
            Common = RichPlugins.Common(rootEl, options);

        return This = {

            init: function() {
                RichPlugins.Utils.init(rootEl, options);
                Common.resize();
                Common.initStyle();
                This.actions();
                RichPlugins.Instances.Lists[collId] = This;
                console.log('RichPlugins List initialized');
            },

            actions: function() {
                window.addEventListener('resize', Common.resize);

                var more = rootEl.getElementsByClassName('rpi-url')[0];
                if (more) {
                    more.onclick = function() {
                        Common.loadNextReviews(options.pagination, function() {
                            let count = parseInt(cntEl.getAttribute('data-count')),
                                offset = parseInt(cntEl.getAttribute('data-offset'));
                            if (offset >= count) {
                                RichPlugins.Utils.rm(more);
                            }
                        });
                        return false;
                    }
                }
            }
        }
    },

    /**
     * Slider lite layout
     */
    Slider: function(rootEl, options) {

        const TIMEOUT_RESIZE = 150,
            TIMEOUT_RESIZE_COLUMN = 50,
            TIMEOUT_SCROLL = 150,

            collId       = rootEl.getAttribute('data-id'),
            cntEl        = rootEl.getElementsByClassName('rpi-cnt')[0],
            contentEl    = rootEl.getElementsByClassName('rpi-content')[0],
            reviewsEl    = contentEl.getElementsByClassName('rpi-cards')[0],
            dotsWrapEl   = contentEl.getElementsByClassName('rpi-dots-wrap')[0],
            dotsEl       = contentEl.getElementsByClassName('rpi-dots')[0],
            //options      = JSON.parse(rootEl.getAttribute('data-opts')),
            sliderOpts   = JSON.parse(cntEl.getAttribute('data-opts')),
            reviewsCount = parseInt(cntEl.getAttribute('data-count'));

        var This            = RichPlugins.Instances.Sliders[collId],
            Common          = RichPlugins.Common(rootEl, options),
            reviewsList     = contentEl.getElementsByClassName('rpi-card'),
            cntElSize       = '',
            currBrPoint     = '',
            resizeTimout    = null,
            swipeAutoTimout = null,
            scrollTimeout   = null,
            wheelTimeout    = null,

            mouseOver   = false,
            btnClickWas = false,

            wheelSpeed  = 0,
            reviewsIdx = 0;

        if (This != null) {
            This.clear();
        }

        return This = {

            init: function() {
                if (Common.isVisible(rootEl)) {
                    setTimeout(function() {
                        RichPlugins.Utils.init(rootEl, options);
                        This.resize();
                        Common.initStyle();
                        This.actions();
                        if (reviewsList.length) {
                            This.swipeAutoStart();
                        }
                    }, 1);
                } else {
                    setTimeout(This.init, 30);
                }

                RichPlugins.Instances.Sliders[collId] = This;
                console.log('RichPlugins slider initialized');
            },

            resize: function(vv) {
                let size = Common.resize(),
                    col = RichPlugins.Utils.getCol();

                if (vv && reviewsEl.scrollLeft != vv * This.reviewWidth()) {
                    //reviewsEl.scrollLeft = vv * This.reviewWidth();
                    reviewsEl.scrollTo(reviewsList[vv].offsetLeft, 0);
                }

                if (reviewsList.length && (cntElSize != size || currBrPoint != col)) {
                    //setTimeout(function() {
                        if (This.canInitDots()) {
                            This.dotsInit();
                            This.dotSwipe(vv, true);
                        }
                        cntElSize = size;
                        currBrPoint = col;
                    //}, TIMEOUT_RESIZE_COLUMN);
                }
            },

            actions: function() {
                if (sliderOpts.mousestop) {
                    This.addMouseEvents();
                }

                window.addEventListener('resize', This.resizeListener);

                if (reviewsEl) {
                    reviewsEl.addEventListener('scroll', This.scrollListener, false);
                    if (sliderOpts.wheelscroll) {
                        contentEl.addEventListener('wheel', This.wheelListener, false);
                    }
                }

                var prev = cntEl.getElementsByClassName('rpi-slider-prev')[0];
                if (prev) {
                    prev.onclick = function(e) {
                        This.btnClick(-1);
                    };
                }

                var next = cntEl.getElementsByClassName('rpi-slider-next')[0];
                if (next) {
                    next.onclick = function(e) {
                        This.btnClick(1);
                    };
                }
            },

            resizeListener: function() {
                var vv = reviewsIdx;
                clearTimeout(resizeTimout);
                resizeTimout = setTimeout(This.resize, TIMEOUT_RESIZE, vv);
            },

            scrollListener: function() {
                clearTimeout(swipeAutoTimout);
                clearTimeout(scrollTimeout);
                scrollTimeout = setTimeout(This.scrollEnd, TIMEOUT_SCROLL);
            },

            wheelListener: function(e) {
                var t = e.target,
                    textEl = t.className.indexOf('rpi-text') > -1 ? t : (t.parentNode.className.indexOf('rpi-text') > -1 ? t.parentNode : null);
                if (textEl && textEl.scrollHeight > textEl.clientHeight) {
                    return true;
                }
                e.preventDefault();
                wheelSpeed++;
                clearTimeout(wheelTimeout);
                wheelTimeout = setTimeout(This.wheelEnd, TIMEOUT_SCROLL, e);
            },

            addMouseEvents: function() {
                cntEl.addEventListener('mouseover', This.mouseOver, false);
                cntEl.addEventListener('mouseleave', This.mouseLeave, false);
            },

            delMouseEvents: function() {
                cntEl.removeEventListener('mouseover', This.mouseOver);
                cntEl.removeEventListener('mouseleave', This.mouseLeave);
            },

            mouseOver: function() {
                mouseOver = 1;
                This.swipeAutoStop();
            },

            mouseLeave: function() {
                mouseOver = 0;
                This.swipeAutoStart();
            },

            btnClick: function(d) {
                This.swipeHand(d * This.swipePerBtn());
            },

            wheelEnd: function(e) {
                This.swipeHand(Math.sign(e.wheelDelta) * wheelSpeed * This.swipeStep());
                wheelSpeed = 0;
            },

            swipeHand: function(step) {
                btnClickWas = true;

                This.loadNextReviews(step);
                This.scroll(step);

                if (sliderOpts.clickstop) {
                    This.swipeAutoStop();
                    This.delMouseEvents();
                }
            },

            scroll: function(steps) {
                //reviewsEl.scrollBy(This.reviewWidth() * steps, 0);

                let newIdx = This.reviewsIdx() + parseInt(steps);

                // Specific case if desync manual clicks (swipe) and scrollTo (went beyond)
                newIdx = newIdx < 0 ? 0 : (newIdx >= reviewsList.length ? reviewsList.length - 1 : newIdx);

                if (newIdx > -1 && newIdx < reviewsList.length) {
                    reviewsEl.scrollTo({
                        left: reviewsList[newIdx].offsetLeft,
                        behavior: 'smooth'
                    });
                    if (This.canInitDots()) {
                        This.dotSwipe(newIdx);
                    }
                }
            },

            scrollEnd: function() {
                // reviewsIdx variable is needed for correctly positioning when resize event called
                reviewsIdx = This.reviewsIdx();

                if (btnClickWas) {
                    btnClickWas = false;
                } else {
                    This.loadNextReviews();
                }

                if ((sliderOpts.mousestop && !mouseOver || !sliderOpts.mousestop) && (sliderOpts.clickstop && !btnClickWas || !sliderOpts.clickstop)) {
                    This.swipeAutoStart();
                }

                if (This.canInitDots()) {
                    This.dotSwipe(undefined, true);
                }
            },

            loadNextReviews: function(step) {
                let reviewsLeft;

                if (step) {
                    reviewsLeft = This.reviewsIdx() + parseInt(step);
                } else if (sliderOpts.hide_dots) {
                    reviewsLeft = This.reviewsIdx();
                } else {
                    let dotAct = dotsEl.getElementsByClassName('active')[0];
                    // If dots enabled get dot active index OR reviews back
                    reviewsLeft = dotAct ? parseInt(dotAct.getAttribute('data-index')) * This.swipePerDot() : This.reviewsIdx();
                }

                let size = This.getAjaxSize(reviewsLeft);
                if (size > 0) {
                    Common.loadNextReviews(size);
                }
            },

            /**
            * This function returns how many reviews should be requested in Ajax call
            *
            * This size is positive only if:
            *
            * 1) Not ALL reviews loaded (reviewsCount > offset)
            * 2) NEAR to offset with some ratio(3) (Math.abs(diff) < 3 * This.swipePerDot())
            * 3) OR Go to FAR dot (diff)
            *
            * reviewsCount - total reviews reviewsCount
            * offset       - how many reviews already loaded
            * reviewsLeft  - total number of requested reviews (newActiveDotIndex * This.swipePerDot())
            * diff         - different between requested and already loaded reviews
            * needsLoad    - how many reviews to request based on pagination
            */
            getAjaxSize: function(reviewsLeft) {
                let size = 0;
                const offset = parseInt(cntEl.getAttribute('data-offset')),
                      pagination = parseInt(options.pagination);

                if (reviewsCount > offset) {

                    let diff = reviewsLeft - offset;

                    if (Math.abs(diff) < 3 * This.swipePerDot()) {
                        size = pagination;

                    } else if (diff) {
                        let needsLoad = Math.ceil(reviewsLeft / pagination) * pagination;
                        size = needsLoad - offset;
                    }
                }

                // In case if AJAX call returns more then total reviews count reviews
                let diffBetweenNextAndTotalReviewsCount = (offset + size) - reviewsCount;
                return diffBetweenNextAndTotalReviewsCount > 0 ? size - diffBetweenNextAndTotalReviewsCount : size;
            },

            dotsInit: function() {
                if (!dotsEl) return;

                let dotsCount = Math.round(reviewsCount / This.swipePerDot());

                dotsEl.innerHTML = '';
                for (let i = 0; i < dotsCount; i++) {
                    let dot = document.createElement('div');
                    dot.className = 'rpi-dot';
                    dot.setAttribute('data-index', i);
                    dot.setAttribute('title', i);
                    dot.onclick = This.dotClick;
                    dotsEl.appendChild(dot);
                }
                This.dotsPadding();
            },

            dotClick: function() {
                let idxNew = parseInt(this.getAttribute('data-index')),
                    dotOld = dotsEl.getElementsByClassName('active')[0],
                    idxOld = parseInt(dotOld.getAttribute('data-index')),
                    idxDiff = Math.abs(idxNew - idxOld);

                This.swipeHand(idxDiff * This.swipePerDot() * Math.sign(idxNew - idxOld));
            },

            dotsPadding: function() {
                let dotsHeight = dotsWrapEl.getBoundingClientRect().height;
                cntEl.style.paddingBottom = dotsHeight + 'px';
            },

            dotSwipe: function(newIdx, fast) {
                let dotIdx = Math.round((newIdx !== undefined ? newIdx : This.reviewsIdx()) / This.swipePerDot());

                // Specific case if desync manual clicks (swipe) and scrollTo (went beyond)
                dotIdx = dotIdx < 0 ? 0 : (dotIdx >= dotsEl.childNodes.length ? dotsEl.childNodes.length - 1 : dotIdx);

                let dotNew = dotsEl.querySelector('.rpi-dot[data-index="' + dotIdx + '"]'),
                    dotOld = dotsEl.getElementsByClassName('active')[0];

                This.dotActivate(dotOld, dotNew);
                This.dotScroll(dotNew, fast);
            },

            dotScroll: function(dotNew, fast) {
                let c = Math.round(dotsEl.scrollWidth / dotsEl.childNodes.length),
                    x = Math.round(dotsEl.offsetWidth / c),
                    z = Math.floor(x / 2);

                if (fast) {
                    dotsEl.scrollTo(dotNew.offsetLeft - z*c, 0);
                } else {
                    dotsEl.scrollTo({
                        left: dotNew.offsetLeft - z*c,
                        behavior: 'smooth'
                    });
                }
            },

            dotActivate: function(dotOld, dotNew) {
                if (dotOld) {
                    dotOld.classList.remove('active');
                    dotOld.classList.remove('s1');
                    if (dotOld.previousSibling) {
                        dotOld.previousSibling.classList.remove('s2');
                        if (dotOld.previousSibling.previousSibling) dotOld.previousSibling.previousSibling.classList.remove('s3');
                    }
                    if (dotOld.nextSibling) {
                        dotOld.nextSibling.classList.remove('s2');
                        if (dotOld.nextSibling.nextSibling) dotOld.nextSibling.nextSibling.classList.remove('s3');
                    }
                }
                if (dotNew) {
                    dotNew.classList.add('active');
                    dotNew.classList.add('s1');
                    if (dotNew.previousSibling) {
                        dotNew.previousSibling.classList.add('s2');
                        if (dotNew.previousSibling.previousSibling) dotNew.previousSibling.previousSibling.classList.add('s3');
                    }
                    if (dotNew.nextSibling) {
                        dotNew.nextSibling.classList.add('s2');
                        if (dotNew.nextSibling.nextSibling) dotNew.nextSibling.nextSibling.classList.add('s3');
                    }
                }
            },

            swipeAuto: function() {
                if (This.isScrollEnd()) {
                    // To return back a reviews count should be subtracted by visible reviews (per view)
                    This.scroll(-(reviewsCount - This.reviewsPerView()));
                } else {
                    // If reviews ahead less then swipe step, use reviews ahead count
                    let step = This.swipeStep() < This.reviewsAhead() ? This.swipeStep() : This.reviewsAhead();
                    This.scroll(step);
                }
                This.swipeAutoStart();
            },

            swipeAutoStart: function() {
                if (sliderOpts.autoplay) {
                    swipeAutoTimout = setTimeout(This.swipeAuto, parseInt(sliderOpts.speed) * 1000);
                }
            },

            swipeAutoStop: function() {
                clearTimeout(swipeAutoTimout);
                if (scrollTimeout) {
                    setTimeout(function() { clearTimeout(scrollTimeout) }, 100);
                }
            },

            isScrollEnd: function() {
                var lastReview = reviewsEl.querySelector('.rpi-card:last-child'),
                    elemRect   = lastReview.getBoundingClientRect(),
                    parentRect = lastReview.parentNode.getBoundingClientRect();

                return (Math.abs(parentRect.left - elemRect.left) < 2 || parentRect.left <= elemRect.left) && elemRect.left < parentRect.right &&
                       (Math.abs(parentRect.right - elemRect.right) < 2 || parentRect.right >= elemRect.right) && elemRect.right > parentRect.left;
            },

            swipeStep: function() {
                return sliderOpts.swipe_step || This.reviewsPerView();
            },

            swipePerBtn: function() {
                return sliderOpts.swipe_per_btn || This.reviewsPerView();
            },

            swipePerDot: function() {
                return sliderOpts.swipe_per_dot || This.reviewsPerView();
            },

            reviewWidth: function() {
                //return reviewsList[0].offsetWidth;
                return Math.round(reviewsEl.scrollWidth / reviewsList.length);
            },

            reviewHeight: function() {
                return reviewsList[0].offsetHeight;
            },

            reviewsPerView: function() {
                return Math.round(reviewsEl.offsetWidth / This.reviewWidth());
            },

            reviewsIdx: function() {
                return Math.round(reviewsEl.scrollLeft / This.reviewWidth());
            },

            reviewsAhead: function() {
                return reviewsList.length - (This.reviewsIdx() + This.reviewsPerView());
            },

            canInitDots: function() {
                return !sliderOpts.hide_dots && This.swipePerDot() > 0;
            },

            clear: function() {
                clearTimeout(resizeTimout);
                clearTimeout(swipeAutoTimout);
                clearTimeout(scrollTimeout);
                clearTimeout(wheelTimeout);
                window.removeEventListener('resize', This.resizeListener);
                reviewsEl.removeEventListener('scroll', This.scrollListener);
                contentEl.removeEventListener('wheel', This.wheelListener);
            }
        }
    },

    init: function(rootEl) {
        rootEl.setAttribute('data-exec', '1');
        RichPlugins.Utils.rm(document.getElementById('rpi-style'));

        let opts = JSON.parse(rootEl.getAttribute('data-opts')),
            layout = RichPlugins.Utils.capit(opts.layout);
        if (layout == 'List' || layout == 'Grid') {
            RichPlugins.List(rootEl, opts).init();
        } else {
            RichPlugins[layout](rootEl, opts).init();
        }
    }

};

document.addEventListener('DOMContentLoaded', function() {
    const els = document.querySelectorAll('.rpi[data-exec=""]');
    for (var i = 0; i < els.length; i++) {
        RichPlugins.init(els[i]);
    }
});